/* eslint-disable @typescript-eslint/no-explicit-any */
import { useCallback, useContext, useEffect, useRef, useState } from 'react';
import { deserializeData, buildUrl } from '@agentuity/frontend';
import { AgentuityContext } from './context';
/**
 * Parse route key into method and path
 */
function parseRouteKey(routeKey) {
    const parts = routeKey.split(' ');
    if (parts.length !== 2) {
        throw new Error(`Invalid route key format: "${routeKey}". Expected "METHOD /path"`);
    }
    return { method: parts[0], path: parts[1] };
}
/**
 * Convert query object to URLSearchParams
 */
function toSearchParams(query) {
    if (!query)
        return undefined;
    if (query instanceof URLSearchParams)
        return query;
    return new URLSearchParams(query);
}
/**
 * Substitute path parameters in a URL path template.
 * E.g., '/users/:id' with { id: '123' } becomes '/users/123'
 */
function substitutePathParams(pathTemplate, pathParams) {
    if (!pathParams)
        return pathTemplate;
    let result = pathTemplate;
    for (const [key, value] of Object.entries(pathParams)) {
        result = result.replace(new RegExp(`:${key}\\??`, 'g'), encodeURIComponent(value));
        result = result.replace(new RegExp(`\\*${key}`, 'g'), encodeURIComponent(value));
    }
    return result;
}
/**
 * Process ReadableStream with delimiter-based parsing
 */
async function processStream(stream, options) {
    const { delimiter, onChunk, onData, onError, mountedRef } = options;
    const reader = stream.getReader();
    const decoder = new TextDecoder();
    let buffer = '';
    try {
        while (true) {
            const { done, value } = await reader.read();
            if (!mountedRef.current) {
                reader.cancel();
                return false;
            }
            if (done)
                break;
            buffer += decoder.decode(value, { stream: true });
            const parts = buffer.split(delimiter);
            buffer = parts.pop() || '';
            for (const part of parts) {
                if (!part.trim())
                    continue;
                try {
                    // Try JSON parsing first, fall back to plain string for text streams
                    let parsed;
                    try {
                        parsed = JSON.parse(part);
                    }
                    catch {
                        // Not valid JSON - treat as plain string
                        parsed = part;
                    }
                    if (onChunk) {
                        parsed = await Promise.resolve(onChunk(parsed));
                    }
                    if (!mountedRef.current) {
                        reader.cancel();
                        return false;
                    }
                    onData(parsed);
                }
                catch (err) {
                    if (!mountedRef.current) {
                        reader.cancel();
                        return false;
                    }
                    const error = err instanceof Error ? err : new Error(String(err));
                    onError(error);
                    return false;
                }
            }
        }
        if (buffer.trim()) {
            try {
                // Try JSON parsing first, fall back to plain string for text streams
                let parsed;
                try {
                    parsed = JSON.parse(buffer);
                }
                catch {
                    // Not valid JSON - treat as plain string
                    parsed = buffer;
                }
                if (onChunk) {
                    parsed = await Promise.resolve(onChunk(parsed));
                }
                if (mountedRef.current) {
                    onData(parsed);
                }
            }
            catch (err) {
                if (!mountedRef.current)
                    return false;
                const error = err instanceof Error ? err : new Error(String(err));
                onError(error);
                return false;
            }
        }
    }
    catch (err) {
        if (!mountedRef.current)
            return false;
        const error = err instanceof Error ? err : new Error(String(err));
        onError(error);
        return false;
    }
    return true;
}
// Implementation signature
export function useAPI(routeOrOptions) {
    // Normalize to options object - use plain object type since we're in the implementation
    const options = typeof routeOrOptions === 'string'
        ? { route: routeOrOptions }
        : routeOrOptions;
    const context = useContext(AgentuityContext);
    const { input, query, headers, enabled, staleTime = 0, refetchInterval, onSuccess, onError, } = options;
    // Extract params safely
    const pathParams = 'params' in options ? options.params : undefined;
    const delimiter = 'delimiter' in options ? (options.delimiter ?? '\n') : '\n';
    const onChunk = 'onChunk' in options ? options.onChunk : undefined;
    if (!context) {
        throw new Error('useAPI must be used within AgentuityProvider');
    }
    // Extract method and path from either route OR {method, path}
    let method;
    let basePath;
    if ('route' in options && options.route) {
        const parsed = parseRouteKey(options.route);
        method = parsed.method;
        basePath = parsed.path;
    }
    else if ('method' in options && 'path' in options && options.method && options.path) {
        method = options.method;
        basePath = options.path;
    }
    else {
        throw new Error('useAPI requires either route OR {method, path}');
    }
    // Substitute path parameters
    const path = substitutePathParams(basePath, pathParams);
    const [data, setData] = useState(undefined);
    const [error, setError] = useState(null);
    const [isLoading, setIsLoading] = useState(false);
    const [isFetching, setIsFetching] = useState(false);
    const [isSuccess, setIsSuccess] = useState(false);
    const [isError, setIsError] = useState(false);
    const lastFetchTimeRef = useRef(0);
    // Track mounted state to prevent state updates after unmount
    const mountedRef = useRef(true);
    useEffect(() => {
        mountedRef.current = true;
        return () => {
            mountedRef.current = false;
        };
    }, []);
    // Use refs to store latest delimiter/onChunk values to avoid stale closures
    // without causing infinite loops in useCallback dependencies
    const delimiterRef = useRef(delimiter);
    const onChunkRef = useRef(onChunk);
    useEffect(() => {
        delimiterRef.current = delimiter;
        onChunkRef.current = onChunk;
    });
    const fetchData = useCallback(async () => {
        if (!mountedRef.current)
            return;
        const now = Date.now();
        const lastFetchTime = lastFetchTimeRef.current;
        // Check if data is still fresh based on last fetch time
        const isFresh = staleTime > 0 && lastFetchTime !== 0 && now - lastFetchTime < staleTime;
        if (isFresh) {
            return;
        }
        setIsFetching(true);
        // isLoading = only for first load (or after reset)
        if (lastFetchTime === 0) {
            setIsLoading(true);
        }
        setError(null);
        setIsError(false);
        try {
            const url = buildUrl(context.baseUrl || '', path, undefined, toSearchParams(query));
            const requestInit = {
                method,
                headers: {
                    'Content-Type': 'application/json',
                    ...(context.authHeader && { Authorization: context.authHeader }),
                    ...headers,
                },
            };
            // Add body for non-GET requests
            if (method !== 'GET' && input !== undefined) {
                requestInit.body = JSON.stringify(input);
            }
            const response = await fetch(url, requestInit);
            if (!response.ok) {
                throw new Error(`HTTP ${response.status}: ${response.statusText}`);
            }
            let responseData;
            // Handle 204 No Content - no response body expected
            if (response.status === 204) {
                responseData = undefined;
            }
            else {
                const contentType = response.headers.get('Content-Type') || '';
                if (contentType.includes('text/event-stream') ||
                    contentType.includes('application/octet-stream')) {
                    if (!response.body) {
                        throw new Error('Response body is null for streaming response');
                    }
                    setData([]);
                    // Track accumulated chunks locally to avoid stale closure
                    const accumulatedChunks = [];
                    const success = await processStream(response.body, {
                        delimiter: delimiterRef.current,
                        onChunk: onChunkRef.current,
                        onData: (chunk) => {
                            if (!mountedRef.current)
                                return;
                            accumulatedChunks.push(chunk);
                            setData([...accumulatedChunks]);
                        },
                        onError: (err) => {
                            if (!mountedRef.current)
                                return;
                            setError(err);
                            setIsError(true);
                            onError?.(err);
                        },
                        mountedRef,
                    });
                    if (!mountedRef.current)
                        return;
                    if (success) {
                        setIsSuccess(true);
                        lastFetchTimeRef.current = Date.now();
                        const finalData = accumulatedChunks;
                        onSuccess?.(finalData);
                    }
                    return;
                }
                else if (contentType.includes('application/json')) {
                    responseData = await response.json();
                }
                else {
                    const text = await response.text();
                    responseData = deserializeData(text);
                }
            }
            if (!mountedRef.current)
                return;
            // Use JSON memoization to prevent re-renders when data hasn't changed
            setData((prev) => {
                const newData = responseData;
                if (prev !== undefined && JSON.stringify(prev) === JSON.stringify(newData)) {
                    return prev;
                }
                return newData;
            });
            setIsSuccess(true);
            lastFetchTimeRef.current = Date.now();
            onSuccess?.(responseData);
        }
        catch (err) {
            if (!mountedRef.current)
                return;
            const error = err instanceof Error ? err : new Error(String(err));
            setError(error);
            setIsError(true);
            onError?.(error);
        }
        finally {
            if (mountedRef.current) {
                setIsLoading(false);
                setIsFetching(false);
            }
        }
    }, [
        context.baseUrl,
        context.authHeader,
        path,
        method,
        input,
        query,
        headers,
        staleTime,
        onSuccess,
        onError,
    ]);
    const reset = useCallback(() => {
        setData(undefined);
        setError(null);
        setIsLoading(false);
        setIsFetching(false);
        setIsSuccess(false);
        setIsError(false);
        lastFetchTimeRef.current = 0;
    }, []);
    // For GET requests: auto-fetch and provide refetch
    if (method === 'GET') {
        const refetch = useCallback(async () => {
            await fetchData();
        }, [fetchData]);
        // Auto-fetch on mount if enabled (default: true for GET)
        const shouldAutoFetch = enabled ?? true;
        useEffect(() => {
            if (shouldAutoFetch) {
                fetchData();
            }
        }, [shouldAutoFetch, fetchData]);
        // Refetch interval
        useEffect(() => {
            if (!refetchInterval || refetchInterval <= 0)
                return;
            const interval = setInterval(() => {
                fetchData();
            }, refetchInterval);
            return () => clearInterval(interval);
        }, [refetchInterval, fetchData]);
        return {
            data,
            error,
            isLoading,
            isSuccess,
            isError,
            isFetching,
            refetch,
            reset,
        };
    }
    // For POST/PUT/PATCH/DELETE: provide invoke method (manual invocation)
    const invoke = useCallback(async (invokeInput) => {
        // Use invokeInput parameter if provided
        const effectiveInput = invokeInput !== undefined ? invokeInput : input;
        setIsFetching(true);
        setIsLoading(true);
        setError(null);
        setIsError(false);
        try {
            const url = buildUrl(context.baseUrl || '', path, undefined, toSearchParams(query));
            const requestInit = {
                method,
                headers: {
                    'Content-Type': 'application/json',
                    ...(context.authHeader && { Authorization: context.authHeader }),
                    ...headers,
                },
            };
            if (effectiveInput !== undefined) {
                requestInit.body = JSON.stringify(effectiveInput);
            }
            const response = await fetch(url, requestInit);
            if (!response.ok) {
                throw new Error(`HTTP ${response.status}: ${response.statusText}`);
            }
            let responseData;
            if (response.status === 204) {
                responseData = undefined;
            }
            else {
                const contentType = response.headers.get('Content-Type') || '';
                if (contentType.includes('text/event-stream') ||
                    contentType.includes('application/octet-stream')) {
                    if (!response.body) {
                        throw new Error('Response body is null for streaming response');
                    }
                    setData([]);
                    // Track accumulated chunks locally to avoid stale closure
                    const accumulatedChunks = [];
                    let streamError = undefined;
                    const success = await processStream(response.body, {
                        delimiter: delimiterRef.current,
                        onChunk: onChunkRef.current,
                        onData: (chunk) => {
                            if (!mountedRef.current)
                                return;
                            accumulatedChunks.push(chunk);
                            setData([...accumulatedChunks]);
                        },
                        onError: (err) => {
                            if (!mountedRef.current)
                                return;
                            streamError = err;
                            setError(err);
                            setIsError(true);
                            onError?.(err);
                        },
                        mountedRef,
                    });
                    if (!mountedRef.current)
                        return accumulatedChunks;
                    if (!success && streamError) {
                        throw streamError;
                    }
                    if (success) {
                        setIsSuccess(true);
                        lastFetchTimeRef.current = Date.now();
                        const finalData = accumulatedChunks;
                        onSuccess?.(finalData);
                        return finalData;
                    }
                    return accumulatedChunks;
                }
                else if (contentType.includes('application/json')) {
                    responseData = await response.json();
                }
                else {
                    const text = await response.text();
                    responseData = deserializeData(text);
                }
            }
            if (!mountedRef.current)
                return responseData;
            setData(responseData);
            setIsSuccess(true);
            lastFetchTimeRef.current = Date.now();
            onSuccess?.(responseData);
            return responseData;
        }
        catch (err) {
            if (!mountedRef.current)
                throw err;
            const error = err instanceof Error ? err : new Error(String(err));
            setError(error);
            setIsError(true);
            onError?.(error);
            throw error;
        }
        finally {
            if (mountedRef.current) {
                setIsLoading(false);
                setIsFetching(false);
            }
        }
    }, [context.baseUrl, context.authHeader, path, method, query, headers, input, onSuccess, onError]);
    return {
        data,
        error,
        isLoading,
        isSuccess,
        isError,
        isFetching,
        invoke,
        reset,
    };
}
//# sourceMappingURL=api.js.map