import { jsx as _jsx } from "react/jsx-runtime";
import { useCallback, useEffect, useRef, useState } from 'react';
import { getAnalytics } from '@agentuity/frontend';
const noopClient = {
    track: () => { },
    identify: () => { },
    flush: () => { },
};
/**
 * Hook for tracking analytics events in React components
 *
 * @example
 * ```tsx
 * function MyComponent() {
 *   const { track, trackClick } = useAnalytics();
 *
 *   useEffect(() => {
 *     track('component_viewed', { name: 'MyComponent' });
 *   }, [track]);
 *
 *   return (
 *     <button onClick={trackClick('button_clicked', { button: 'submit' })}>
 *       Submit
 *     </button>
 *   );
 * }
 * ```
 */
export function useAnalytics() {
    const clientRef = useRef(null);
    const [isClient, setIsClient] = useState(false);
    // SSR guard - only initialize on client
    useEffect(() => {
        setIsClient(true);
        if (!clientRef.current) {
            clientRef.current = getAnalytics();
        }
    }, []);
    // Use no-op client during SSR, real client after hydration
    const client = isClient && clientRef.current ? clientRef.current : noopClient;
    const track = useCallback((eventName, properties) => {
        client.track(eventName, properties);
    }, [client]);
    const trackClick = useCallback((eventName, properties) => {
        return (_event) => {
            client.track(eventName, properties);
        };
    }, [client]);
    const identify = useCallback((userId, traits) => {
        client.identify(userId, traits);
    }, [client]);
    const flush = useCallback(() => {
        client.flush();
    }, [client]);
    return {
        track,
        trackClick,
        identify,
        flush,
        ready: isClient && clientRef.current !== null,
    };
}
/**
 * Hook to track an event when a component mounts
 *
 * @example
 * ```tsx
 * function ProductPage({ productId }: { productId: string }) {
 *   useTrackOnMount({
 *     eventName: 'product_viewed',
 *     properties: { productId }
 *   });
 *
 *   return <div>Product {productId}</div>;
 * }
 * ```
 */
export function useTrackOnMount(options) {
    const { eventName, properties, once = true } = options;
    const trackedRef = useRef(false);
    const { track } = useAnalytics();
    useEffect(() => {
        if (once && trackedRef.current) {
            return;
        }
        track(eventName, properties);
        trackedRef.current = true;
    }, [eventName, once, properties, track]);
}
/**
 * Higher-order component for tracking page views
 *
 * @example
 * ```tsx
 * const TrackedHomePage = withPageTracking(HomePage, 'home');
 * ```
 */
export function withPageTracking(Component, pageName) {
    const displayName = Component.displayName || Component.name || 'Component';
    function TrackedComponent(props) {
        useTrackOnMount({
            eventName: 'page_view',
            properties: {
                page: pageName || displayName,
            },
        });
        return _jsx(Component, { ...props });
    }
    TrackedComponent.displayName = `withPageTracking(${displayName})`;
    return TrackedComponent;
}
//# sourceMappingURL=analytics.js.map