/**
 * Log Collector for clean build logs
 *
 * Provides a mechanism to collect clean, non-animated log output
 * for streaming to external services (like Pulse) while keeping
 * animated TUI output for the user's terminal.
 *
 * Usage:
 * - Set AGENTUITY_CLEAN_LOGS_FILE env var to a file path
 * - TUI components call appendLog() for final state messages
 * - Logs are written to the file for the parent process to read
 */
import { appendFileSync, writeFileSync } from 'node:fs';
/**
 * Get the clean logs file path from environment
 */
function getCleanLogsFile() {
    return process.env.AGENTUITY_CLEAN_LOGS_FILE;
}
/**
 * Disable log collection (called on write errors to prevent repeated failures)
 */
function disableLogCollection(error) {
    console.debug('Log collection disabled due to write error: %s', error);
    delete process.env.AGENTUITY_CLEAN_LOGS_FILE;
}
/**
 * Check if log collection is enabled (via environment variable)
 */
export function isLogCollectionEnabled() {
    return !!getCleanLogsFile();
}
/**
 * Initialize the clean logs file (clears any existing content)
 */
export function initCleanLogsFile(filePath) {
    try {
        writeFileSync(filePath, '');
        process.env.AGENTUITY_CLEAN_LOGS_FILE = filePath;
    }
    catch (err) {
        console.debug('Failed to initialize clean logs file: %s', err);
    }
}
/**
 * Append a clean log line (no ANSI codes, no animation)
 * Only appends if collection is enabled
 */
export function appendLog(message) {
    const file = getCleanLogsFile();
    if (file) {
        try {
            appendFileSync(file, message + '\n');
        }
        catch (err) {
            disableLogCollection(err);
        }
    }
}
/**
 * Append multiple log lines
 */
export function appendLogs(messages) {
    const file = getCleanLogsFile();
    if (file) {
        try {
            appendFileSync(file, messages.join('\n') + '\n');
        }
        catch (err) {
            disableLogCollection(err);
        }
    }
}
//# sourceMappingURL=log-collector.js.map